/*____________________________________________________________________________
	Copyright (C) 2000 Networks Associates Technology, Inc.
	All rights reserved.

	$Id: pgpContext.c,v 1.50.2.3 2001/06/27 15:07:19 jdool Exp $
____________________________________________________________________________*/
#include <stdio.h>

#include "pgpErrors.h"
#include "pgpMem.h"
#include "pgpPFLPriv.h"
#include "pgpSDKPriv.h"

#include "pgpContext.h"
#include "pgpKeys.h"
#include "pgpEnv.h"
#include "pgpRandomPoolPriv.h"
#include "pgpRandomContext.h"
#include "pgpRandomX9_17.h"
#include "pgpRnd.h"
#include "pgpRndSeed.h"
#include "pgpUtilitiesPriv.h"
#include "pgpMacFileMapping.h"
#include "bn.h"
#include "pgpMemoryMgrPriv.h"
#include "pgpMemPool.h"
#include "pgpTimeBomb.h"
#include "pgpPassCach.h"
#include "pgpKeyPriv.h"
#include "pgpThreads.h"

#if PGP_MACINTOSH
#include "PGPsdkDriverAPI.h"
#include "PGPsdkBackEndAPI.h"
#endif

#if PGP_WIN32
#include <windows.h>
#include "PGPsdkDriver.h"
#include "PGPsdkBackEndAPI.h"
#endif

#if PGP_UNIX
#include "pgpRPCBackEndAPI.h"
#endif

/* Global variable to keep track of all contexts in use */
static PGPContextRef pgpsdkFirstContext;

/* Global to record keydb update notification callback */
static PGPNotificationHandlerProc		sNotificationProc;
static PGPUserValue						sNotificationUserValue;

/* Global mutex lock defined in utilities/pgpPassCach.c */
extern PGPMutex_t gPassCacheMutex;		

enum
{
	kPGPContextMagic	= 0x436e7478	/* 'Cntx' */
};

struct PGPContext
{
	PGPContextRef		nextContext;
	PGPMemoryMgrRef		memoryMgr;
	PGPMemoryMgrRef		memPoolMgr;
	MemPool				memPool;

	PGPUInt32			magic;				/* Always kPGPContextMagic */
	PGPUInt32			clientAPIVersion;	/* version client compiled against */
	PGPUserValue		userValue;			/* Client storage */
	
	PGPEnv		   *	pgpEnvironment;		/* State for within library */
	
	PGPRandomContext *	randomPoolContext;	/* ctxt around global rand pool*/
	PGPRandomContext *	randomPoolX9_17;	/* uses 'randomPoolContext' */
	PGPRandomContext *	dummyRandomContext;

	PGPKeyDBRef			keyDB;				/* List of keydbs */
	PGPConnectRef		connect_ref;		/* Back end ref for front end */
	struct PGPCacheHeader *passphraseCache;	/* Passphrase cache header */
	PGPUInt32			currentThread;		/* thread owning context */
	PGPBoolean			updateNeeded;		/* Out of sync with backend */
};

typedef struct PGPContext	PGPContext;



/*____________________________________________________________________________
	This function validates a context data structure.
____________________________________________________________________________*/

	PGPBoolean
pgpContextIsValid(PGPContextRef context)
{
	return( context != NULL &&
			context->magic == kPGPContextMagic );
}


/*____________________________________________________________________________
	Allocate a new PGPContext using custom memory allocators.
	This function es exported for the client library and should never be
	called by the shared library.
____________________________________________________________________________*/
	static PGPError
FinishInitContext( PGPContextRef  context )
{
	PGPError	err = kPGPError_NoErr;
		
	err = pgpenvCreate( context, &context->pgpEnvironment );

	/* Create a mempool and prime it with a small allocation */
	memPoolInit( context, &context->memPool );
	memPoolAlloc( &context->memPool, sizeof(void *), sizeof(void *) );
	context->memPoolMgr = pgpMemPoolMemoryMgr( &context->memPool );
	
#if PGP_WIN32
	context->currentThread = GetCurrentThreadId();
#endif

	return( err );
}

/*____________________________________________________________________________
	Allocate a new PGPContext using custom memory allocators.
	This function es exported for the client library and should never be
	called by the shared library.
____________________________________________________________________________*/

	PGPError
pgpNewContextCustomInternal(
	PGPCustomContextInfo const *	custom,
	PGPContextRef*					newContext)
{
	PGPError			err = kPGPError_NoErr;
	PGPContextRef		context	= NULL;

	PGPValidatePtr( newContext );
	*newContext	= NULL;
	PGPValidatePtr( custom );
	/*	Note: if the struct grows, then
		sizeof( PGPCustomContextInfo ) will need to be a fixed number */ 
	PGPValidateParam( PGPMemoryMgrIsValid( custom->memoryMgr ) );
	
	if ( ! pgpsdkIsInited() )
	{
		return( kPGPError_ImproperInitialization );
	}

	if ( custom->sdkAPIVersion != kPGPsdkAPIVersion )
	{
		return( kPGPError_IncompatibleAPI );
	}
	
#if PGP_TIME_BOMB
	if ( pgpTimeBombHasExpired() )
	{
		*newContext	= NULL;
		return( kPGPError_FeatureNotAvailable );
	}
#endif

	context	= (PGPContext *)PGPNewData( custom->memoryMgr,
				sizeof( PGPContext ), kPGPMemoryMgrFlags_Clear );
	
	if( IsntPGPError( err ) )
	{
		context->memoryMgr			= custom->memoryMgr;
		context->magic				= kPGPContextMagic;
		context->clientAPIVersion	= custom->sdkAPIVersion;
		
		err	= FinishInitContext( context );

		if( IsPGPError( err ) )
		{
			PGPFreeContext( context );
			context = kInvalidPGPContextRef;
		} else {
			context->nextContext		= pgpsdkFirstContext;
			pgpsdkFirstContext			= context;
		}
	}
	else
	{
		err = kPGPError_OutOfMemory;
	}
	
	*newContext	= context;
	
	return( err );
}

/*____________________________________________________________________________
	Allocate a new PGPContext using the default PGPsdk memory allocators.
____________________________________________________________________________*/

	PGPError 
PGPNewContext(
	PGPUInt32		clientAPIVersion,
	PGPContextRef *	newContext)
{
	PGPCustomContextInfo	custom;
	PGPError				err	= kPGPError_NoErr;
	
	PGPValidatePtr( newContext );
	*newContext	= NULL;

	pgpEnterPGPErrorFunction();

	pgpClearMemory( &custom, sizeof( custom ) );
	custom.sdkAPIVersion = clientAPIVersion;
	
	err	= PGPNewMemoryMgr( 0, &custom.memoryMgr );
	if ( IsntPGPError( err ) )
	{
		err	= pgpNewContextCustomInternal( &custom, newContext );
	}
	
	return( err );
}


/*____________________________________________________________________________
	Allocate a new custom PGPContext.
____________________________________________________________________________*/
	PGPError
PGPNewContextCustom(
	PGPCustomContextInfo const *		custom,
	PGPContextRef 					*newContext)
{
	PGPError	err	= kPGPError_NoErr;
	
	PGPValidatePtr( newContext );
	*newContext	= kInvalidPGPContextRef;
	PGPValidatePtr( custom );
	
	pgpEnterPGPErrorFunction();

	err	= pgpNewContextCustomInternal( custom, newContext );
	
	return( err );
}


	static void
sDestroyContext( PGPContextRef	context )
{
	if( IsntNull( context->memPoolMgr ) )
	{
		PGPFreeMemoryMgr( context->memPoolMgr );
	}
	memPoolEmpty( &context->memPool );

	if( IsntNull( context->pgpEnvironment ) )
	{
		pgpenvDestroy( context->pgpEnvironment );
		context->pgpEnvironment	= NULL;
	}
	
	if ( IsntNull( context->randomPoolContext ) )
	{
		pgpRandomDestroy( context->randomPoolContext );
		context->randomPoolContext	= NULL;
	}
	
	if ( IsntNull( context->randomPoolX9_17 ) )
	{
		pgpRandomDestroy( context->randomPoolX9_17 );
		context->randomPoolX9_17	= NULL;
	}
	
	if ( IsntNull( context->dummyRandomContext ) )
	{
		pgpRandomDestroy( context->dummyRandomContext );
		context->dummyRandomContext	= NULL;
	}
	if( IsntNull( context->passphraseCache ) )
	{
		PGPPurgePassphraseCache( context );
	}
}

/*____________________________________________________________________________
	Delete an existing PGPContext and all resources associated with it.
____________________________________________________________________________*/

	PGPError 
PGPFreeContext(PGPContextRef context)
{
	PGPError		err = kPGPError_NoErr;
	PGPMemoryMgrRef	mgr;
	
	PGPValidateContext( context );
	
	pgpEnterPGPErrorFunction();

	/* Unlink context from static list */
	if( context == pgpsdkFirstContext )
	{
		pgpsdkFirstContext = context->nextContext;
	} else {
		PGPContextRef prevContext = pgpsdkFirstContext;
		while( prevContext->nextContext != context )
		{
			pgpAssert( IsntNull( prevContext->nextContext ) );
			prevContext = prevContext->nextContext;
		}
		prevContext->nextContext = context->nextContext;
	}

	/* Get current time into rand pool */
	PGPGlobalRandomPoolAddKeystroke( PGPGetTime() );
	err	= pgpSaveGlobalRandomPool( );
	
	sDestroyContext( context );
	
	mgr	= context->memoryMgr;
	PGPFreeData( context );
	PGPFreeMemoryMgr( mgr );
	
	/* Tell backend */

	return( err );
}



/*____________________________________________________________________________
	Allocate a block of memory using the allocator stored in a PGPContext.
____________________________________________________________________________*/

	void *
pgpContextMemAlloc(
	PGPContextRef 		context,
	PGPSize 			requestSize,
	PGPMemoryMgrFlags	flags)
{
	void	*	allocation = NULL;
	PGPSize		allocationSize	= requestSize;
	
	pgpAssert( pgpContextIsValid( context ) );
	
	allocation = PGPNewData( context->memoryMgr,
					allocationSize, flags );
	
	return( allocation );
}

/*____________________________________________________________________________
	Same as pgpContextMemAlloc(), but copies data into newly allocated block.
____________________________________________________________________________*/

	void *
PGPContextMemAllocCopy(
	PGPContextRef 		context,
	PGPSize 			requestSize,
	PGPMemoryMgrFlags	flags,
	const void *		dataToCopy )
{
	void *	allocation;
	
	pgpAssert( pgpContextIsValid( context ) );
	
	allocation = PGPNewData( context->memoryMgr,
					requestSize, flags & ~kPGPMemoryMgrFlags_Clear );
	if ( IsntNull( allocation ) )
	{
		pgpCopyMemory( dataToCopy, allocation, requestSize );
	}
	
	return( allocation );
}


/*____________________________________________________________________________
	Allocate a block of memory using the allocator stored in a PGPContext.
____________________________________________________________________________*/

	PGPError
pgpContextMemRealloc(
	PGPContextRef 		context,
	void				**allocation,
	PGPSize 			requestSize,
	PGPMemoryMgrFlags	flags)
{
	PGPError	err	= kPGPError_NoErr;
	
	err = PGPReallocData( context->memoryMgr,
			allocation, requestSize, flags );
	return( err );
}

/*____________________________________________________________________________
	Free a block of memory using the deallocator stored in a PGPContext.
____________________________________________________________________________*/

	PGPError
pgpContextMemFree(
	PGPContextRef 	context,
	void * 			allocation)
{
	PGPError	err;
	
	PGPValidateContext( context );
	(void)context;
	
	err = PGPFreeData( allocation );
	
	return( err );
}

/*____________________________________________________________________________
    "Lightweight" versions of memory allocators above, based on mempools.
    Do cutback on the mempool to free all that has been allocated.
____________________________________________________________________________*/
	void *
pgpContextMemPoolAlloc(
	PGPContextRef 		context,
	PGPSize 			requestSize,
	PGPMemoryMgrFlags	flags)
{
	void	*	allocation = NULL;
	PGPSize		allocationSize	= requestSize;
	
	pgpAssert( pgpContextIsValid( context ) );
	
	allocation = PGPNewData( context->memPoolMgr, allocationSize, flags );
	
	return( allocation );
}

	PGPError
pgpContextMemPoolRealloc(
	PGPContextRef 		context,
	void				**allocation,
	PGPSize 			requestSize,
	PGPMemoryMgrFlags	flags)
{
	PGPError	err	= kPGPError_NoErr;
	
	pgpAssert( pgpContextIsValid( context ) );

	err = PGPReallocData( context->memPoolMgr, allocation, requestSize,
						  flags );
	return( err );
}

/* Use this to free all data allocated since doing
 * cutpool = *pgpPeekContextMemPool( context );
 */
	void
pgpContextMemPoolCutBack(
	PGPContextRef 		context,
	MemPool const		*cutpool)
{
	pgpAssert( pgpContextIsValid( context ) );

	memPoolCutBack( &context->memPool, cutpool );
}

	
/*____________________________________________________________________________
    UserValue and other miscellaneous functions
____________________________________________________________________________*/
	PGPError 
PGPGetContextUserValue(
	PGPContextRef 	context,
	PGPUserValue 	*userValue)
{
	PGPError	err = kPGPError_NoErr;
	
	PGPValidatePtr( userValue );
	*userValue	= NULL;
	PGPValidateContext( context );
	
	pgpEnterPGPErrorFunction();

	*userValue = context->userValue;
	
	return( err );
}


	PGPMemoryMgrRef
PGPPeekContextMemoryMgr( PGPContextRef context )
{
	if ( ! pgpContextIsValid( context ) )
		return( NULL );
		
	pgpEnterZeroFunction();

	return( context->memoryMgr );
}


	MemPool *
pgpPeekContextMemPool( PGPContextRef context )
{
	if ( ! pgpContextIsValid( context ) )
		return( NULL );
		
	return( &context->memPool );
}

	PGPMemoryMgrRef
pgpPeekContextMemPoolMgr( PGPContextRef context )
{
	if ( ! pgpContextIsValid( context ) )
		return( NULL );
		
	pgpEnterZeroFunction();

	return( context->memPoolMgr );
}

/* Iterate over all context values.  If called with null, return first */
	PGPContextRef
pgpContextNextContext( PGPContextRef ctx )
{
	if( IsNull( ctx ) )
		return pgpsdkFirstContext;

	pgpAssert( pgpContextIsValid( ctx ) );

	return ctx->nextContext;
}


	PGPError 
PGPSetContextUserValue(
	PGPContextRef 	context,
	PGPUserValue 	userValue)
{
	PGPError	err = kPGPError_NoErr;
	
	PGPValidateContext( context );
	
	pgpEnterPGPErrorFunction();

	context->userValue = userValue;
	
	return( err );
}


	PGPConnectRef
pgpContextGetConnectRef(
	PGPContextRef 	context )
{
#if PGP_MACINTOSH
	ProcessSerialNumber	processSerial;
	
	context;
	GetCurrentProcess(&processSerial);
	return (PGPConnectRef)processSerial.lowLongOfPSN;
#else
	pgpAssert( pgpContextIsValid( context ) );
	return context->connect_ref;
#endif
}

	void
pgpContextSetConnectRef(
	PGPContextRef 	context,
	PGPConnectRef	connect_ref )
{
#if PGP_MACINTOSH
	context;	connect_ref;
	pgpDebugMsg( "Connection Refs cannot be set manually on the Macintosh" );
#else
	pgpAssert( pgpContextIsValid( context ) );
	context->connect_ref = connect_ref;
#endif
}

	PGPUInt32
pgpContextGetThreadID(
	PGPContextRef	context )
{
	pgpAssert( pgpContextIsValid( context ) );
	return context->currentThread;
}

	struct PGPCacheHeader *
pgpContextGetPassphraseCache(
	PGPContextRef 	context )
{
	pgpAssert( pgpContextIsValid( context ) );
	return context->passphraseCache;
}

	void
pgpContextSetPassphraseCache(
	PGPContextRef 			context,
	struct PGPCacheHeader *	passphraseCache )
{
	pgpAssert( pgpContextIsValid( context ) );
	context->passphraseCache = passphraseCache;
}


	PGPEnv *
pgpContextGetEnvironment(PGPContextRef context)
{
	pgpAssert( pgpContextIsValid( context ) );

	return( context->pgpEnvironment );
}

	PGPKeyDBRef
pgpContextGetFirstKeyDB(PGPContextRef context)
{
	pgpAssert( pgpContextIsValid( context ) );

	return( context->keyDB );
}

	void
pgpContextSetFirstKeyDB(PGPContextRef context, PGPKeyDBRef keyDB)
{
	pgpAssert( pgpContextIsValid( context ) );

	context->keyDB = keyDB;
}

	PGPBoolean
pgpContextGetUpdateNeeded( PGPContextRef context )
{
	pgpAssert( pgpContextIsValid( context ) );

	return context->updateNeeded;
}

	void
pgpContextSetUpdateNeeded( PGPContextRef context, PGPBoolean isNeeded )
{
	pgpAssert( pgpContextIsValid( context ) );

	context->updateNeeded = isNeeded;
}





	static PGPError
sCreateRandomContext( 
	PGPContextRef			context,
	PGPRandomVTBL const *	vtbl,
	PGPRandomContext **		outRandom )
{
	PGPError			err	= kPGPError_NoErr;
	PGPRandomContext *	newRandom	= NULL;
	
	pgpAssert( pgpContextIsValid( context ) );
	
		
	newRandom	= (PGPRandomContext *)
		pgpContextMemAlloc( context, sizeof( *newRandom ),
		kPGPMemoryMgrFlags_Clear );
	if ( IsntNull( newRandom ) )
	{
		newRandom->context	= context;
		newRandom->vtbl		= vtbl;
		newRandom->priv		= NULL;
		newRandom->destroy	= NULL;
	}
	else
	{
		err	= kPGPError_OutOfMemory;
	}

	*outRandom	= newRandom;
	return( err );
}



	PGPRandomContext *
pgpContextGetGlobalPoolRandomContext(PGPContextRef context)
{
	pgpAssert( pgpContextIsValid( context ) );
	
	if ( IsNull( context->randomPoolContext ) )
	{
		PGPRandomContext *	randomContext	= NULL;
		
		(void)sCreateRandomContext( context,
			pgpGetGlobalRandomPoolVTBL(), &randomContext );
		context->randomPoolContext	= randomContext;
		
		if ( IsntNull( context->randomPoolContext ) )
		{
			/*
			* in case there is a problem in reading the random pool,
			* feed in some amount of "random" data to avoid a totally
			* predictible pool
			*/
			pgpRandomCollectEntropy( randomContext );
			pgpRandomCollectEntropy( randomContext );
			pgpRandomCollectEntropy( randomContext );
			pgpRandomCollectEntropy( randomContext );
		}
	}

	return( context->randomPoolContext );
}



	PGPRandomContext *
pgpContextGetDummyRandomContext(PGPContextRef context)
{
	pgpAssert( pgpContextIsValid( context ) );
	
	if ( IsNull( context->dummyRandomContext ) )
	{
		(void)sCreateRandomContext( context,
			pgpGetGlobalDummyRandomPoolVTBL(),
			&context->dummyRandomContext );
	}

	return( context->dummyRandomContext );
}



	PGPRandomContext *
pgpContextGetX9_17RandomContext(PGPContextRef context)
{
	pgpAssert( pgpContextIsValid( context ) );
	
	if ( IsNull( context->randomPoolX9_17 ) )
	{
		context->randomPoolX9_17	= pgpRandomCreate( context );
	}

	return( context->randomPoolX9_17 );
}


/*____________________________________________________________________________
    Fill the user-specified buffer with random values from the global
    random pool.  If we were not able to initialize the random pool
    we return kPGPError_OutOfEntropy.
____________________________________________________________________________*/
	PGPError
PGPContextGetRandomBytes(
	PGPContextRef	context,
	void *			buf,
	PGPSize			len
	)
{
	PGPRandomContext *	randomContext;

	PGPValidatePtr( buf );
	PGPValidateParam( pgpContextIsValid( context ) );

	pgpEnterPGPErrorFunction();

	randomContext = pgpContextGetX9_17RandomContext( context );
	if( IsNull( randomContext )
		|| !PGPGlobalRandomPoolHasMinimumEntropy() )
	{
		return kPGPError_OutOfEntropy;
	}

	pgpRandomGetBytes( randomContext, buf, len );
	
	return kPGPError_NoErr;
}

/*____________________________________________________________________________
    Reserve random bytes in the front end copy (if it exists).  This will
	ensure that we have data for later random requests.  Return the number
    of bytes reserved in total so far.  To just learn that value, call this
    function with 0.
____________________________________________________________________________*/
	PGPUInt32
PGPContextReserveRandomBytes(
	PGPContextRef	context,
	PGPUInt32		minsize
	)
{
	PGPRandomContext *	randomContext;
	PGPUInt32 reserved;

	PGPValidateParam( pgpContextIsValid( context ) );

	pgpEnterPGPErrorFunction();

	randomContext = pgpContextGetX9_17RandomContext( context );
	if( IsNull( randomContext )
		|| !PGPGlobalRandomPoolHasMinimumEntropy() )
	{
		return 0;
	}

	reserved = pgpRandomReserveBytes( randomContext, minsize );
	
	return reserved;
}

#if PGP_WIN32

	static PGPError
pgpSDKInitPlatform(PGPFlags options)
{
	PGPError	err = kPGPError_NoErr, tempErr;
	
	tempErr = pgpSDKInitDriver();
//	pgpAssertNoErr( tempErr );

	/*
	 * The SDK creates a FrontEndThread whether or not it
	 * is being run in "local" mode.  When there is no SDK
	 * service, this thread is responsible for expiring
	 * the passphrase cache.  When there is an SDK service,
	 * the thread receives our key-change notifications.
	 */
	if (options & kPGPFlags_ForceLocalExecution) {
		if (!(options & kPGPFlags_SuppressCacheThread))
			pgpInitFrontEndThread(TRUE);
	}
	else {
		if (!(options & kPGPFlags_SuppressCacheThread))
			pgpInitFrontEndThread(FALSE);
		err = pgpSDKInitBackEnd( kPGPsdkBackEndAPIVersion);
	}

	return err;
}

#elif PGP_MACINTOSH	/* ][ */

	static PGPError
pgpSDKInitPlatform(PGPFlags options)
{
	PGPError	err = kPGPError_NoErr;
	
	(void) options;
	
#if PGPSDK_FRONTEND
	/*
	** Failure to load the driver is not fatal.
	** May not be enough memory is sys heap
	*/

	err = pgpSDKInitDriver();
	pgpAssertNoErr( err );
	
	err = pgpSDKInitBackEnd( kPGPsdkBackEndAPIVersion);
	if( IsntPGPError( err ) ) {
		err = pgpConnect_backRPC();
	}
#endif
	
	return err;
}

#else	/* ][ */
	static PGPError
pgpSDKInitPlatform(PGPFlags options)
{
	PGPError err = kPGPError_NoErr;

	if (!(options & kPGPFlags_ForceLocalExecution))
		err = pgpSDKInitBackEnd(kPGPsdkBackEndAPIVersion);

	return err;
}
#endif	/* ] */

static PGPUInt32	sInitedCount		= 0;
static PGPBoolean	sFreeMemoryMgrList	= FALSE;

	PGPError
PGPsdkInit(PGPFlags options)
{
	PGPError	err	= kPGPError_NoErr;
	
	if ( sInitedCount == 0 )
	{
	#if PGPSDK_FRONTEND
		pgpLeaksBeginSession( "PGPsdk" );
	#else
		pgpLeaksBeginSession( "PGPsdkBackEnd" );
	#endif
	
		if( IsNull( PGPGetDefaultMemoryMgr() ) )
		{
			PGPMemoryMgrRef	memoryMgr;
			
			sFreeMemoryMgrList = TRUE;
			
			err = PGPNewMemoryMgr( 0, &memoryMgr );
			pgpAssertNoErr( err );
			if( IsntPGPError( err ) )
			{
				err = PGPSetDefaultMemoryMgr( memoryMgr );
				pgpAssertNoErr( err );
			}
		}
		
		if( IsntPGPError( err ) )
		{
			err	= pgpSDKInitPlatform( options );
			pgpAssertNoErr( err );
		}

		if( IsntPGPError( err ) )
		{
			err	= pgpInitMacBinaryMappings();
			pgpAssertNoErr( err );
		}
		
		if( IsntPGPError( err ) )
		{
			bnInit();
		}

		if( IsntPGPError( err ) )
		{
			err = pgpDefaultRandSeedFile ();
		}

		PGPMutexCreate(&gPassCacheMutex, NULL);
	}
	
	if ( IsntPGPError( err ) )
	{
		++sInitedCount;
	}
	
	return( err );
}


	PGPBoolean
pgpsdkIsInited( void )
{
	return( sInitedCount != 0 );
}

	PGPError
pgpForceSDKCleanup( void )
{
	PGPError	err	= kPGPError_NoErr;
	
	if ( sInitedCount != 0 )
	{
		sInitedCount	= 1;
	}
	
	err	= PGPsdkCleanup();
	return( err );
}

#if PGP_MACINTOSH	/* [ */

	static void
pgpSDKCleanupPlatform(void)
{
#if PGPSDK_FRONTEND
	pgpDisconnect_backRPC();
	pgpSDKCleanupBackEnd();
#endif
}

#elif PGP_UNIX /* ][ */

	static void
pgpSDKCleanupPlatform(void)
{
	pgpDisconnect_backRPC();
}

#else	/* ][ */

	static void
pgpSDKCleanupPlatform(void)
{
	pgpKillFrontEndThread();
	pgpDisconnect_backRPC();
}

#endif	/* ] */

	PGPError
PGPsdkCleanup( void )
{
	PGPError	err	= kPGPError_NoErr;
	
	pgpAssert( sInitedCount != 0 );
	if ( sInitedCount != 0 )
	{
		--sInitedCount;
		if ( sInitedCount == 0 )
		{
			pgpUnloadTCL();

			pgpDisposeMacBinaryMappings();
			
			pgpSDKCleanupPlatform();

			pgpCleanupRandSeedFile();

			PGPMutexDestroy(&gPassCacheMutex);

			if( sFreeMemoryMgrList )
				pgpFreeDefaultMemoryMgrList();
			
			pgpLeaksEndSession();
		}
	}
	else
	{
		err	= kPGPError_BadParams;
	}
	
	return( err );
}

	PGPError
PGPSetNotificationCallback( PGPNotificationHandlerProc proc,
	PGPUserValue userValue )
{
	sNotificationProc = proc;
	sNotificationUserValue = userValue;
	return kPGPError_NoErr;
}

/*
 * Call from the asynchronous code which is told that a notify update is
 * ready.
 */
	void
pgpCallNotificationCallback ()
{
	(void)pgpSetBackendUpdateNeeded();
	if( IsntNull( sNotificationProc ) )
		(*sNotificationProc) ( sNotificationUserValue );
}


/*__Editor_settings____

	Local Variables:
	tab-width: 4
	End:
	vi: ts=4 sw=4
	vim: si
_____________________*/
